{ config, pkgs, ... }:

{
  #############  Basisting  #############
  nixpkgs.config.allowUnfree = true;
  boot.loader.systemd-boot.enable = false;
  boot.loader.timeout = 10;
  boot.loader.grub = {
    enable = true;
    efiSupport = true;
    useOSProber = true;
    devices = [ "nodev" ];
    efiInstallAsRemovable = true;
    timeoutStyle = "menu";
    default = 0;

    configurationLimit = 5;  # 👈 Vis de 3 nyeste NixOS-generationer
  };

  
  #############  NVIDIA  ################
  boot.kernelModules = [
    "nvidia" "nvidia_modeset" "nvidia_uvm" "nvidia_drm"
    "xpad"  # Xbox controller support
  ];
  boot.kernelParams = [ "nvidia-drm.modeset=1" ];

  hardware.nvidia = {
    modesetting.enable = true;
    open = false;
    nvidiaSettings = true;
    package = config.boot.kernelPackages.nvidiaPackages.stable;
  };

  hardware.graphics.enable = true;
  hardware.uinput.enable = true;

  ##############  Time  ################
  time.timeZone = "Europe/Copenhagen";

  #############  Filesystemer  ##########
  fileSystems."/" = {
    device = "/dev/disk/by-uuid/31a7ded4-79e9-4a69-9cce-d759156ba625";
    fsType = "ext4";
  };
  fileSystems."/boot" = {
    device = "/dev/disk/by-uuid/3903-F13F";
    fsType = "vfat";
  };

  #############  Display-manager  #######
  services.xserver = {
    enable = true;
    videoDrivers = [ "nvidia" ];
  };
  services.displayManager.gdm.enable = true; # GDM Wayland/X fallback

  #############  Hyprland  ##############
  programs.hyprland = {
    enable = true;
    package = pkgs.hyprland;
  };

  #############  Lyd, net, ssh  #########
  services.pipewire = {
    enable = true;
    alsa.enable = true;
    alsa.support32Bit = true;
    pulse.enable = true;
    jack.enable = true;
  };

  services.pulseaudio.enable = false;

  # Firewall configuration for mDNS and Localsend
  networking.firewall = {
    enable = true;
    allowedUDPPorts = [ 
      5353  # mDNS
      53317 # Localsend default port
    ];
    allowedTCPPorts = [
      53317 # Localsend default port
    ];
  };

  # Enable Avahi for mDNS/Bonjour discovery (needed for Localsend)
  services.avahi = {
    enable = true;
    nssmdns4 = true;
    publish = {
      enable = true;
      addresses = true;
      domain = true;
      hinfo = true;
      userServices = true;
      workstation = true;
    };
  };

  services.mullvad-vpn.enable = true;

  services.openssh = {
    enable = true;
    openFirewall = true;
    settings = {
      PermitRootLogin = "no";
      PasswordAuthentication = true;
    };
  };

  #############  Bruger  ################
  users.users.clemens = {
    isNormalUser = true;
    extraGroups = [ "wheel" "networkmanager" "input" ];
    shell = pkgs.zsh;
  };
  programs.zsh.enable = true;

  #############  Fonte & pakker  ########
  fonts.packages = with pkgs; [
    noto-fonts
    noto-fonts-emoji
  ];

  environment.systemPackages = with pkgs; [
    qt6.qtwayland
    egl-wayland
    git
    adwaita-icon-theme
    steam
    libGL
    glxinfo 
    mesa
    pulseaudio
    pavucontrol
    mono
    ntfs3g
    mullvad-vpn
    sunshine
    xdg-desktop-portal
    xdg-desktop-portal-wlr
    cifs-utils 
    os-prober
    nodejs_latest
    flatpak
    input-remapper
    linuxPackages.xpadneo  # Xbox controller driver
    sg3_utils
    libdvdcss
    eject
    wine
    winetricks
  ];

  # Enable Flatpak support
  services.flatpak.enable = true;

  services.sunshine = {
    enable = true;
    autoStart = false;
    capSysAdmin = true;
    openFirewall = true;
  };

  programs.steam = {
    enable = true;
    remotePlay.openFirewall = true;
    dedicatedServer.openFirewall = true;
  };

  services.input-remapper = {
    enable = true;
    enableUdevRules = true;
  };


  #############  Wayland + NVIDIA vars  ##
  environment.sessionVariables = {
    NIXOS_OZONE_WL = "1";
    __GLX_VENDOR_LIBRARY_NAME = "nvidia";
    WLR_NO_HARDWARE_CURSORS = "1";
    QT_QPA_PLATFORM = "xcb";
    WLR_ENABLE_VIRTUAL_KEYBOARD = "1";
  };

  # nix garbage collection
  nix.gc = {
    automatic = true;
    dates = "weekly";
    options = "--delete-older-than 7d";
  };

  fileSystems."/mnt/windows-system" = {
    device = "/dev/disk/by-uuid/166CBCDF6CBCBB37";
    fsType = "ntfs-3g";
    options = [ "defaults" "uid=1000" "gid=100" ];
  };

  fileSystems."/mnt/windows-data" = {
    device = "/dev/disk/by-uuid/1E02CB9402CB6EF9";
    fsType = "ntfs-3g";
    options = [ "defaults" "uid=1000" "gid=100" ];
  };

  fileSystems."/home/clemens/<removed>" = {
    device = "//<removed>";
    fsType = "cifs";
    options = [ "guest" "uid=1000" "gid=1000" "_netdev" "nofail" ];
  };

  services.dbus.enable = true;
  services.udisks2.enable = true;

  environment.etc."fuse.conf".text = ''
    user_allow_other
  '';

  nix.settings.experimental-features = [ "nix-command" "flakes" ];

  system.stateVersion = "24.05";

  networking.hostName = "rogue";
}
