{ config, pkgs, ... }:
let
  # Webapp helper function
  webapp = name: url: icon: category: {
    name = name;
    comment = "${name} web application";
    exec = "chromium --app=${url} --ozone-platform=wayland --enable-features=UseOzonePlatform";
    icon = icon;
    terminal = false;
    categories = [ category ];
  };
in
{
  # User and home directory
  home.username = "clemens";
  home.homeDirectory = "/home/clemens";

  # Import modular app configs
  imports = [
    ./apps/rofi.nix      # Rofi launcher (themed)
    ./apps/kitty.nix     # Kitty terminal (themed)
    ./apps/zsh.nix       # Zsh shell config
    ./apps/waybar.nix    # Waybar status bar (themed)
    ./apps/p10k.nix      # Powerlevel10k prompt
    ./apps/swayimg.nix   # Swayimg image viewer config
    ./apps/gtk.nix       # GTK, icon, and cursor theming
    ./apps/mako.nix      # Mako notification daemon (themed)
    ./apps/wlogout.nix   # Wlogout config
    ./apps/spotify.nix   # Spotify via Flatpak
    ./apps/neovim.nix    # Neovim configuration
  ];

  # Main user packages, grouped by purpose
  home.packages = with pkgs; [
    # Core desktop & Hyprland
    waybar            # Status bar for Wayland
    hyprpaper         # Wallpaper daemon for Hyprland
    hyprlock          # Lock screen for Hyprland
    mako              # Notification daemon (themed)
    swayidle          # Idle management (screen lock, etc.)
    grim              # Screenshot tool (Wayland)
    hyprshot          # Screenshot tool (Hyprland)
    slurp             # Region selection for screenshots
    swayimg           # Image viewer (Wayland)
    wayland-utils     # Utilities for Wayland (e.g., wl-copy, wl-paste)
    cliphist          # Clipboard manager for Wayland
    wlogout           # Logout utility for Wayland
    ydotool           # Command line tool to trigger mouse events
    wf-recorder       # Screen recording for Wayland

    # Theming & session
    libnotify         # Notification support
    xdg-user-dirs     # XDG user directory management
    hyprcursor        # Official Hyprland cursor utilities
    input-remapper    # Key/mouse/gamepad remapping GUI

    # Fonts
    pkgs.nerd-fonts.jetbrains-mono  # JetBrains Mono Nerd Font for icons
    pkgs.nerd-fonts.fira-code       # Fira Code Nerd Font for icons

    # Shell & terminal
    zsh-autosuggestions # Zsh plugin: autosuggestions
    zsh-syntax-highlighting # Zsh plugin: syntax highlighting
    zsh-powerlevel10k # Zsh theme (prompt)
    autojump          # Directory jumping in shell
    htop              # System monitor
    btop              # Modern system monitor
    wget              # Download utility
    unzip             # Archive extraction
    sshfs             # SSH Filesystem (mount remote directories)
    wlrctl            # Wayland input device control
    ncdu              # Disk usage analyzer
    

    # Applications
    firefox           # Web browser
    vscode            # Visual Studio Code
    gh                # GitHub CLI
    gh-copilot        # GitHub Copilot CLI
    keepassxc         # Password manager
    # spotify         # Music streaming - using Flatpak instead
    youtube-music     # YouTube Music web app
    discord           # Chat/voice
    sioyek            # PDF viewer
    lite-xl           # Lightweight text editor
    protontricks      # Proton/Wine helper for games
    # jetbrains.rider   # JetBrains Rider IDE
    vlc               # Media player
    chromium          # Alternative web browser
    fastfetch         # System information tool
    localsend         # Local file sharing tool
    pinta             # Image editor
    signal-desktop    # Encrypted messaging app
    remmina           # Remote desktop client
    _1password        # Password manager (CLI)
    _1password-gui    # 1Password GUI client
    handbrake         # Video converter
    dvdbackup         # DVD ripping tool
    ffmpeg            # Multimedia processing tool
    asciidoctor       # Document converter

    # Games
    iagno             # Iagno game (GNOME's Othello)
    prismlauncher     # Game launcher for Prism Launcher (Minecraft, etc.)
    dhewm3            # Doom 3 game engine

    # Development
    mono              # .NET/Mono runtime and SDK

    # Gaming
    gamemode          # Game performance tuning
    wineWowPackages.stable # Wine for running Windows applications
    lutris            # Game launcher

    # File manager + dependencies
    xfce.thunar       # File manager
    xfce.thunar-volman # Removable media support for Thunar
    gvfs              # Virtual filesystem support (mounting, trash, etc.)
    xfce.tumbler        # thumbnail previews for Thunar
    ffmpegthumbnailer   # for video previews
    poppler_utils       # for PDF previews
    gdk-pixbuf          # for image previews (PNG, JPEG, etc.)


    # Misc system
    libsecret         # Secret storage (for apps like VSCode)
    wl-clipboard      # Clipboard utilities for Wayland
    networkmanager    # Network management (includes nmcli)
    playerctl         # Media player control for keyboard shortcuts
    input-remapper    # Key/mouse/gamepad remapping tool


    # LaTeX packages
    (texliveSmall.withPackages (ps: with ps; [
      acmart
      libertine
      latexmk
      natbib
      booktabs
      collection-latexrecommended
      collection-latexextra

      fontspec      # kræver LuaLaTeX
      geometry
      titling
      collection-luatex
      hyperref
      breakcites
      enumitem
      xcolor
      tcolorbox
      pgf
    ]))
    texlab

    # Pandoc
    pandoc
  ];

  # Session/environment variables for Wayland/Hyprland
  home.sessionVariables = {
    XDG_CURRENT_DESKTOP = "Hyprland";
    XDG_SESSION_DESKTOP = "Hyprland";
    XDG_SESSION_TYPE = "wayland";
    GDK_BACKEND = "wayland,x11";
    WEBRTC_PIPEWIRE_DEVICE = "1"; # Fix for screen sharing in browsers
    # Force dark mode for Qt applications
    QT_STYLE_OVERRIDE = "adwaita-dark";
    # Add Flatpak apps to application search path
    XDG_DATA_DIRS = "$XDG_DATA_DIRS:$HOME/.local/share/flatpak/exports/share";
    TAVILY_API_KEY= "<REMOVED>"; #  Tavily API key for Copilot web search in Neovim
  };

  # Theme variables for GTK, cursor, and desktop
  xdg.configFile."environment.d/00-theme.conf".text = ''
    GTK_THEME=catppuccin-mocha-lavender-standard
    XDG_CURRENT_DESKTOP=Hyprland
    XDG_SESSION_DESKTOP=Hyprland
  '';


  # Webapps using helper function
  xdg.desktopEntries = {
    linkwarden = webapp "Linkwarden" "https://links.multiverset.net" "web-browser" "Network";
    appleNotes = webapp "Notes" "https://www.icloud.com/notes/" "web-browser" "Office";
    iCloud = webapp "iCloud" "https://www.icloud.com/" "web-browser" "Office";
    youtube = webapp "YouTube" "https://youtube.com" "video" "AudioVideo";
    chatgpt = webapp "ChatGPT" "https://chat.openai.com" "robot" "Development";
    mastodon = webapp "Mastodon" "https://social.multiverset.net" "social" "Network";
    overleaf = webapp "Overleaf" "https://overleaf.com" "text-editor" "Office";
    bluesky = webapp "BlueSky" "https://bsky.app" "social" "Network";
    aumail = webapp "AU Mail" "https://webmail.au.dk" "mail" "Network";
    gmail = webapp "Gmail" "https://mail.google.com" "mail" "Network";
    synology = webapp "Synology" "https://multiverset.quickconnect.to" "network-server" "Network";
  };


 
  
 systemd.user.services."input-remapper-autoload" = {
   Unit = {
     Description = "Input Remapper autoload presets";
     After = [ "graphical-session.target" "input-remapper.service" ]; # vent på session + systemd-dæmon
     Wants = [ "input-remapper.service" ];
     PartOf = [ "graphical-session.target" ];
   };
   Service = {
     Type = "oneshot";
     ExecStart = "${pkgs.input-remapper}/bin/input-remapper-control --command autoload";
   };
   Install = {
     WantedBy = [ "graphical-session.target" ];
   };
 };
  # Enable Zsh shell
  programs.zsh.enable = true;

  # Hyprland config (modularized)
  home.file.".config/hypr/hyprland.conf" = {
    source = ./config/hyprland.conf;
  };

  # Hyprpaper wallpaper config
  home.file.".config/hypr/hyprpaper.conf".text = ''
    preload = /home/clemens/Billeder/megadrive.jpg
   wallpaper = DP-1,/home/clemens/Billeder/megadrive.jpg
  '';

  # SSHFS service to mount NAS drive
  systemd.user.services.sshfs-nas-drive = {
    Unit = {
      Description = "Mount NAS ~/Drive via SSHFS";
      After = [ "network-online.target" ];
    };
    Service = {
      ExecStart = "${pkgs.sshfs}/bin/sshfs -o IdentityFile=<removed>,port=<removed> clemens@<removed>:/homes/clemens/Drive /home/clemens/Drive";
      ExecStop = "${pkgs.fuse}/bin/fusermount -u /home/clemens/Drive";
      Restart = "on-failure";
      KillMode = "none";
    };
    Install = {
      WantedBy = [ "default.target" ];
    };
  };

  # Font configuration
  fonts.fontconfig.enable = true;

  # Home Manager state version
  home.stateVersion = "24.05";
}
